/*
 * Decompiled with CFR 0.152.
 */
import java.io.BufferedWriter;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

public class UnifiedLogger {
    private static volatile UnifiedLogger instance;
    private final Logger slf4jLogger;
    private Socket socket;
    private PrintWriter writer;
    private final String logstashHost;
    private final String numSerie;
    private final int logstashPort;
    private ScheduledExecutorService scheduledExecutor;
    private boolean isEnabled = true;
    private static final int MAX_RETRY_COUNT = 3;
    private static final int BATCH_SIZE = 50;
    private static final long PROCESS_INTERVAL = 300000L;
    private static final long MAX_LOG_AGE = 604800000L;
    private LogDatabaseHelper dbHelper;
    private static final int CONNECTION_TIMEOUT = 50000;
    private static final int READ_TIMEOUT = 3000;
    private final ExecutorService logExecutor;
    private int consecutiveTimeouts = 0;
    private static final int MAX_CONSECUTIVE_TIMEOUTS = 3;
    private static final int SEND_TIMEOUT = 5000;

    private UnifiedLogger(Builder builder) {
        this.slf4jLogger = LoggerFactory.getLogger(builder.loggerName);
        this.logstashHost = builder.logstashHost;
        this.logstashPort = builder.logstashPort;
        this.scheduledExecutor = Executors.newScheduledThreadPool(2);
        this.dbHelper = new LogDatabaseHelper();
        this.logExecutor = Executors.newSingleThreadExecutor();
        this.numSerie = builder.numSerie;
        this.setupReconnection();
        this.setupPendingLogsProcessor();
    }

    private void setupReconnection() {
        this.scheduledExecutor.scheduleWithFixedDelay(() -> {
            if (!this.isSocketConnected()) {
                this.connectToLogstash();
            }
        }, 0L, 60L, TimeUnit.SECONDS);
    }

    private void setupPendingLogsProcessor() {
        this.scheduledExecutor.scheduleWithFixedDelay(() -> this.processPendingLogs(), 300000L, 300000L, TimeUnit.MILLISECONDS);
    }

    private synchronized void connectToLogstash() {
        try {
            if (!this.isSocketConnected()) {
                this.socket = new Socket();
                this.socket.setSoTimeout(3000);
                this.socket.setTcpNoDelay(true);
                this.socket.setKeepAlive(true);
                this.socket.connect(new InetSocketAddress(this.logstashHost, this.logstashPort), 50000);
                this.writer = new PrintWriter((Writer)new BufferedWriter(new OutputStreamWriter(this.socket.getOutputStream(), StandardCharsets.UTF_8)), true);
                System.out.println("UnifiedLogger: Connected to Logstash");
                this.consecutiveTimeouts = 0;
            }
        }
        catch (Exception e) {
            System.err.println("UnifiedLogger: Error connecting to Logstash - " + e.getMessage());
            this.closeConnection();
        }
    }

    private boolean isSocketConnected() {
        return this.socket != null && !this.socket.isClosed() && this.socket.isConnected();
    }

    private synchronized void closeConnection() {
        try {
            if (this.writer != null) {
                this.writer.close();
            }
            if (this.socket != null) {
                this.socket.close();
            }
        }
        catch (Exception e) {
            System.err.println("UnifiedLogger: Error closing connection - " + e.getMessage());
        }
        finally {
            this.writer = null;
            this.socket = null;
        }
    }

    public static UnifiedLogger getInstance() {
        if (instance == null) {
            throw new IllegalStateException("UnifiedLogger must be initialized first");
        }
        return instance;
    }

    public void debug(String tag, String message) {
        this.debug(tag, message, null);
    }

    public void debug(String tag, String message, Map<String, String> metadata) {
        if (!this.isEnabled) {
            return;
        }
        this.log("DEBUG", tag, message, metadata);
    }

    public void info(String tag, String message) {
        this.info(tag, message, null);
    }

    public void info(String tag, String message, Map<String, String> metadata) {
        if (!this.isEnabled) {
            return;
        }
        this.log("INFO", tag, message, metadata);
    }

    public void warning(String tag, String message) {
        this.warning(tag, message, null);
    }

    public void warning(String tag, String message, Map<String, String> metadata) {
        if (!this.isEnabled) {
            return;
        }
        this.log("WARNING", tag, message, metadata);
    }

    public void error(String tag, String message) {
        this.log("ERROR", tag, message, null);
    }

    public void error(String tag, String message, Map<String, String> metadata) {
        if (!this.isEnabled) {
            return;
        }
        this.log("ERROR", tag, message, metadata);
    }

    public void error(String tag, String message, Throwable throwable) {
        HashMap<String, String> metadata = new HashMap<String, String>();
        metadata.put("errorClass", throwable.getClass().getName());
        metadata.put("stackTrace", this.getStackTraceString(throwable));
        metadata.put("errorMessage", throwable.getMessage() != null ? throwable.getMessage() : "No message");
        this.log("ERROR", tag, message, metadata);
    }

    private String getStackTraceString(Throwable throwable) {
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        throwable.printStackTrace(pw);
        return sw.toString();
    }

    private synchronized void log(String level, String tag, String message, Map<String, String> metadata) {
        try {
            String logMessage;
            this.setLoggingContext(tag, metadata);
            switch (level) {
                case "DEBUG": {
                    this.slf4jLogger.debug(message);
                    break;
                }
                case "INFO": {
                    this.slf4jLogger.info(message);
                    break;
                }
                case "WARNING": {
                    this.slf4jLogger.warn(message);
                    break;
                }
                case "ERROR": {
                    this.slf4jLogger.error(message);
                }
            }
            MDC.clear();
            try {
                Map<String, String> enhancedMetadata = this.enrichMetadata(metadata);
                JSONObject document = new JSONObject();
                document.put("timestamp", System.currentTimeMillis());
                document.put("level", level);
                document.put("tag", tag);
                document.put("message", message);
                document.put("metadata", new JSONObject(enhancedMetadata));
                document.put("app_type", "java_desktop");
                document.put("device_id", Main.Globals.numSerie);
                JSONObject wrapper = new JSONObject();
                wrapper.put("document", document);
                logMessage = wrapper.toString();
            }
            catch (Exception e) {
                System.err.println("UnifiedLogger: Error creating log message - " + e.getMessage());
                return;
            }
            CompletableFuture.runAsync(() -> {
                try {
                    if (!this.sendToLogstash(logMessage)) {
                        this.dbHelper.saveLog(logMessage);
                    }
                }
                catch (Exception e) {
                    System.err.println("UnifiedLogger: Error in log thread - " + e.getMessage());
                    this.dbHelper.saveLog(logMessage);
                }
            });
        }
        catch (Exception e) {
            System.err.println("UnifiedLogger: Error logging message - " + e.getMessage());
        }
    }

    private synchronized void processPendingLogs() {
        if (!this.isSocketConnected()) {
            this.connectToLogstash();
        }
        if (this.isSocketConnected()) {
            List<PendingLog> pendingLogs = this.dbHelper.getPendingLogs(50);
            for (PendingLog log : pendingLogs) {
                try {
                    if (System.currentTimeMillis() - log.getTimestamp() > 604800000L) {
                        this.dbHelper.deleteLog(log.getId());
                        continue;
                    }
                    if (this.sendToLogstash(log.getMessage())) {
                        this.dbHelper.deleteLog(log.getId());
                        continue;
                    }
                    log.incrementRetryCount();
                    if (log.getRetryCount() >= 3) {
                        this.dbHelper.deleteLog(log.getId());
                        continue;
                    }
                    this.dbHelper.updateRetryCount(log.getId(), log.getRetryCount());
                }
                catch (Exception e) {
                    System.err.println("UnifiedLogger: Error processing pending log: " + log.getId() + " - " + e.getMessage());
                }
            }
        }
    }

    private synchronized boolean sendToLogstash(String logMessage) {
        if (!this.isSocketConnected()) {
            return false;
        }
        System.out.println("Enviando log: " + logMessage);
        try {
            Future<Boolean> future = this.logExecutor.submit(() -> {
                try {
                    if (this.writer != null) {
                        this.writer.println(logMessage);
                        this.writer.flush();
                        if (this.writer.checkError()) {
                            this.closeConnection();
                            return false;
                        }
                        this.consecutiveTimeouts = 0;
                        return true;
                    }
                }
                catch (Exception e) {
                    System.err.println("UnifiedLogger: Error in send thread - " + e.getMessage());
                    this.closeConnection();
                }
                return false;
            });
            return future.get(5000L, TimeUnit.MILLISECONDS);
        }
        catch (TimeoutException e) {
            System.err.println("UnifiedLogger: Send timeout");
            ++this.consecutiveTimeouts;
            if (this.consecutiveTimeouts >= 3) {
                System.out.println("UnifiedLogger: Max consecutive timeouts reached, forcing reconnection");
                this.closeConnection();
                this.connectToLogstash();
                this.consecutiveTimeouts = 0;
            }
            return false;
        }
        catch (Exception e) {
            System.err.println("UnifiedLogger: Error sending log to Logstash - " + e.getMessage());
            this.closeConnection();
            return false;
        }
    }

    private void setLoggingContext(String tag, Map<String, String> metadata) {
        MDC.put("tag", tag);
        if (metadata != null) {
            for (Map.Entry<String, String> entry : metadata.entrySet()) {
                MDC.put(entry.getKey(), entry.getValue());
            }
        }
    }

    private Map<String, String> enrichMetadata(Map<String, String> metadata) {
        HashMap<String, String> enriched = new HashMap<String, String>();
        if (metadata != null) {
            enriched.putAll(metadata);
        }
        enriched.put("osName", System.getProperty("os.name"));
        enriched.put("osVersion", System.getProperty("os.version"));
        enriched.put("osArch", System.getProperty("os.arch"));
        enriched.put("javaVersion", System.getProperty("java.version"));
        enriched.put("javaVendor", System.getProperty("java.vendor"));
        try {
            enriched.put("hostName", InetAddress.getLocalHost().getHostName());
        }
        catch (Exception e) {
            enriched.put("hostName", "unknown");
        }
        Runtime runtime = Runtime.getRuntime();
        enriched.put("totalMemory", String.valueOf(runtime.totalMemory()));
        enriched.put("freeMemory", String.valueOf(runtime.freeMemory()));
        enriched.put("maxMemory", String.valueOf(runtime.maxMemory()));
        enriched.put("serialNumber", Main.Globals.numSerie);
        try {
            String appVersion = this.getClass().getPackage().getImplementationVersion();
            enriched.put("appVersion", appVersion != null ? appVersion : "unknown");
        }
        catch (Exception e) {
            enriched.put("appVersion", "unknown");
        }
        return enriched;
    }

    public void shutdown() {
        try {
            this.isEnabled = false;
            this.closeConnection();
            this.scheduledExecutor.shutdown();
            this.logExecutor.shutdown();
            try {
                if (!this.scheduledExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.scheduledExecutor.shutdownNow();
                }
                if (!this.logExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.logExecutor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.scheduledExecutor.shutdownNow();
                this.logExecutor.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
        catch (Exception e) {
            System.err.println("UnifiedLogger: Error during shutdown - " + e.getMessage());
        }
    }

    public void enable() {
        this.isEnabled = true;
    }

    public void disable() {
        this.isEnabled = false;
    }

    public static class Builder {
        public String numSerie;
        private String logstashHost;
        private int logstashPort = 5000;
        private String loggerName = "AppLogger";

        public Builder logstashHost(String host) {
            this.logstashHost = host;
            return this;
        }

        public Builder logstashPort(int port) {
            this.logstashPort = port;
            return this;
        }

        public Builder loggerName(String name) {
            this.loggerName = name;
            return this;
        }

        public Builder numSerie(String numSerie) {
            this.numSerie = numSerie;
            return this;
        }

        public UnifiedLogger build() {
            UnifiedLogger logger;
            if (this.logstashHost == null) {
                throw new IllegalStateException("Logstash host must be set");
            }
            instance = logger = new UnifiedLogger(this);
            return logger;
        }
    }
}

